package positronic.ai.learner.maip;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;

public class Maip
{
	private Map<Metrizable,Metrizable> backing;

	/**
	 * Sets the lower limit on the size of the Maip's backing map.
	 */
	private int minPopulation;

	/**
	 * The positive number which defines the desired probability of successful
	 * performance by the <tt>Learner</tt> object. Note: in practice, there is no
	 * point in specifying quality values of less than or equal to 0.5, or greater
	 * than or equal to 1.0. Specifying a positive value less than 0.5 is
	 * equivalent to specifying 0.5, and specifying a value greater than 1.0 is
	 * practically equivalent to specifying 1.0.
	 */
	private double quality;

	/**
	 * 
	 * The positive number defining successful performance by the learner object.
	 * That is, if the distance between the correct answer and the learner's answer
	 * is not greater than <code>tolerance</code>, then the learner's performance
	 * in this case is considered successful. Otherwise, its performance is
	 * considered unsuccessful.
	 */
	private double tolerance;

	/**
	 * Constructs a new instance of the <tt>Learner</tt> class with
	 * <tt>tolerance=.01</tt> and <tt>quality=.99</tt>.
	 */
	public Maip()
	{
		this(.1,.9);
	}

	/**
	 * Constructs a new instance of the <tt>Learner</tt> class with specified
	 * values of <tt>tolerance</tt> and <tt>quality</tt>.
	 */
	public Maip(double tolerance, double quality)
	{
		backing=new HashMap<Metrizable,Metrizable>();
		this.tolerance=tolerance;
		this.quality=quality;
	}

	/**
	 * Prints the underlying HashSet to <tt>System.out</tt>.
	 */
	public void print()
	{
		print(System.out);
	}
	
	/**
	 * Prints the underlying HashSet to a <tt>PrintStream</tt>.
	 */
	public void print(PrintStream ps)
	{
		for(Metrizable p : backing.keySet())
			ps.println(p+"\t->\t"+backing.get(p));
	}

	public Metrizable get(Metrizable p)
	{
		Metrizable b=this.nearestPoint(p);
		return backing.get(b);
	}

	public Map<Metrizable, Metrizable> getBacking() 
	{
		return backing;
	}
	
	public int getMinPopulation() 
	{
		return minPopulation;
	}

	public double getQuality() 
	{
		return quality;
	}

	public double getTolerance() 
	{
		return tolerance;
	}

	/**
	 * Returns the <tt>Metrizable</tt> key object in <tt>Maip</tt> 
	 * which is closest to <tt>domainPoint</tt>.
	 */
	public Metrizable nearestPoint(Metrizable domainPoint)
	{
		Metrizable nearestPoint=null;
		double leastDistance=Double.POSITIVE_INFINITY;
		for(Metrizable point : backing.keySet())
		{
			double distancePointToCurrent=domainPoint.distance(point);
			if(distancePointToCurrent<leastDistance)
			{
				nearestPoint=point;
				leastDistance=distancePointToCurrent;
			}
		}
		return nearestPoint;
	}

	public Metrizable put(Metrizable d, Metrizable r)
	{
		if(backing.size()==0 || backing.size()<minPopulation)
		{
			return backing.put(d, r);
		}
		
		Metrizable res = this.nearestPoint(d);
		
		if(r.distance(backing.get(res))>this.getTolerance())
		{
			backing.put(d, r);
			return r;
		}
		else
			if(Math.random()<=1./this.getQuality()-1.)
			{
				backing.remove(res);
			}
		return null;
	}

	public void setMinPopulation(int minPopulation) 
	{
		this.minPopulation = minPopulation;
	}

	public void setQuality(double quality) 
	{
		this.quality = quality;
	}

	public void setTolerance(double tolerance) 
	{
		this.tolerance = tolerance;
	}
	
	public int size() 
	{
		return backing.size();
	}
}
